﻿using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;

public class GameAudioController : MonoBehaviour
{
    public AudioClip BackgroundAudioClip;   //The background audio that plays on loop
    public int LowerAudioPlayTime;  //Lower bounds
    public int UpperAudioPlayTime;  //Upper bounds
    public bool ForcePlay;

    [SerializeField]
    public AmbientAudio[] AmbientAudios;

    private int _AudioPlayTime = -1;    //The time stamp at which to next play some audio
    private float _ElapsedPlayTime = 0.0f;  //The elapsed time since we last played some audio
    private AmbientAudio _LastPlayed;   //The ambient audio we played last time

    /// <summary>
    /// Generates a new time to play audio at between the lower and upper bounds of the play time
    /// </summary>
    private void GetNextAudioPlayTime()
    {
        System.Random rand = new System.Random(Guid.NewGuid().GetHashCode());
        _AudioPlayTime = rand.Next(LowerAudioPlayTime, UpperAudioPlayTime);
    }

    private void PlayAmbientAudio()
    {
        bool playedAudio = false;

        //Loop through all the ambient audio components
        foreach(AmbientAudio ambientAudio in AmbientAudios)
        {
            //If this file isn't the one we've played previously and we should play the file
            if(_LastPlayed != ambientAudio && ambientAudio.ShouldPlay())
            {
                //Play it, update the last played, set the played flag and break from the loop
                ambientAudio.PlayAmbientAudio();
                _LastPlayed = ambientAudio;
                playedAudio = true;
                break;
            }
        }

        if(!playedAudio && ForcePlay)
        {
            //If we've not played an audio file but we're being forced to, pick one at random and play it
            System.Random rand = new System.Random(Guid.NewGuid().GetHashCode());
            int index = rand.Next(0, AmbientAudios.Length - 1);
            AmbientAudios[index].PlayAmbientAudio();
        }
    }
    
    void Start()
    {
        //Begin playing the background audio on loop, subscribe to game state changed
        AudioManager.Instance.PlayMusic(BackgroundAudioClip);
        GameManager.Instance.GameStateChanged += OnGameStateChanged;
    }

    private void OnGameStateChanged(object sender, GameManager.GameStateChangedEventArgs e)
    {
        if (AudioManager.Instance != null)
        {
            if (e.NewState == Constants.GameState.Paused)
            {
                //Reduce music
                AudioManager.Instance.SetMusicVolume(AudioManager.Instance.GetMusicVolume() * 0.2f);
            }

            else
            {
                //Amplify music
                AudioManager.Instance.SetMusicVolume(AudioManager.Instance.GetMusicVolume() * 5.0f);
            }
        }
    }
    
    void Update()
    {
        if(_AudioPlayTime <= -1)    //Safety/sentinel check
        {
            GetNextAudioPlayTime();
        }

        _ElapsedPlayTime += (Time.deltaTime * 1000.0f);

        if(_ElapsedPlayTime >= _AudioPlayTime)
        {
            //Time to play another audio file - play it, reset the elapsed time and get the next time to play
            PlayAmbientAudio();
            _ElapsedPlayTime = 0.0f;
            GetNextAudioPlayTime();
        }
    }
}
